'use client'

import { useEffect, useState } from 'react'
import { useParams, useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Suscripcion, Cliente } from '@/lib/types'
import Link from 'next/link'
import { format } from 'date-fns'
import Modal from '../../components/Modal'

export default function SuscripcionDetailPage() {
  const params = useParams()
  const router = useRouter()
  const [suscripcion, setSuscripcion] = useState<Suscripcion | null>(null)
  const [cliente, setCliente] = useState<Cliente | null>(null)
  const [loading, setLoading] = useState(true)
  const [eliminando, setEliminando] = useState(false)
  const [modalConfirmar, setModalConfirmar] = useState(false)
  const [modalMensaje, setModalMensaje] = useState<{ isOpen: boolean; mensaje: string; tipo: 'success' | 'error' }>({
    isOpen: false,
    mensaje: '',
    tipo: 'success'
  })

  useEffect(() => {
    if (params.id) {
      fetchSuscripcion(params.id as string)
    }
  }, [params.id])

  async function fetchSuscripcion(id: string) {
    try {
      const { data, error } = await supabase
        .from('suscripciones')
        .select(`
          *,
          cliente:clientes(*)
        `)
        .eq('id', id)
        .single()

      if (error) throw error
      setSuscripcion(data)
      setCliente(data.cliente as Cliente)
    } catch (error) {
      console.error('Error fetching suscripcion:', error)
    } finally {
      setLoading(false)
    }
  }

  function abrirModalEliminar() {
    setModalConfirmar(true)
  }

  async function eliminarSuscripcion() {
    if (!suscripcion) return

    setEliminando(true)
    setModalConfirmar(false)

    try {
      const { error } = await supabase
        .from('suscripciones')
        .delete()
        .eq('id', suscripcion.id)

      if (error) throw error

      setModalMensaje({
        isOpen: true,
        mensaje: 'Suscripción eliminada exitosamente',
        tipo: 'success'
      })
      setTimeout(() => {
        router.push('/suscripciones')
      }, 1500)
    } catch (error: any) {
      console.error('Error deleting suscripcion:', error)
      setModalMensaje({
        isOpen: true,
        mensaje: `Error al eliminar la suscripción: ${error.message}`,
        tipo: 'error'
      })
    } finally {
      setEliminando(false)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  if (!suscripcion) {
    return (
      <div className="text-center p-8">
        <p>Suscripción no encontrada</p>
        <Link href="/suscripciones" className="text-blue-600 hover:underline">Volver a suscripciones</Link>
      </div>
    )
  }

  const vencida = suscripcion.fecha_fin && new Date(suscripcion.fecha_fin) < new Date()
  const porVencer = suscripcion.fecha_fin && (() => {
    const fin = new Date(suscripcion.fecha_fin)
    const hoy = new Date()
    const diasRestantes = Math.ceil((fin.getTime() - hoy.getTime()) / (1000 * 60 * 60 * 24))
    return diasRestantes <= 7 && diasRestantes > 0
  })()

  return (
    <div className="max-w-4xl mx-auto">
      <div className="mb-4 flex justify-between items-center">
        <Link href="/suscripciones" className="text-blue-600 hover:underline">← Volver a suscripciones</Link>
        <div className="flex gap-2">
          <Link
            href={`/suscripciones/${suscripcion.id}/editar`}
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition text-sm"
          >
            ✏️ Editar
          </Link>
          <button
            onClick={abrirModalEliminar}
            disabled={eliminando}
            className="bg-red-600 text-white px-4 py-2 rounded hover:bg-red-700 disabled:opacity-50 transition text-sm"
          >
            {eliminando ? 'Eliminando...' : '🗑️ Eliminar'}
          </button>
        </div>
      </div>
      
      <div className={`bg-white p-6 rounded-lg shadow mb-6 ${
        vencida ? 'border-l-4 border-red-500' :
        porVencer ? 'border-l-4 border-yellow-500' :
        suscripcion.estado === 'activa' ? 'border-l-4 border-green-500' :
        'border-l-4 border-gray-500'
      }`}>
        <div className="flex justify-between items-start mb-6">
          <div>
            <h1 className="text-3xl font-bold mb-2">Suscripción #{suscripcion.id.slice(0, 8)}</h1>
            <Link href={`/clientes/${suscripcion.cliente_id}`} className="text-blue-600 hover:underline">
              👤 {cliente?.nombre || 'Cliente desconocido'}
            </Link>
          </div>
          <span className={`px-3 py-1 rounded text-sm font-medium ${
            vencida ? 'bg-red-100 text-red-800' :
            porVencer ? 'bg-yellow-100 text-yellow-800' :
            suscripcion.estado === 'activa' ? 'bg-green-100 text-green-800' :
            'bg-gray-100 text-gray-800'
          }`}>
            {vencida ? 'Vencida' :
             porVencer ? 'Por vencer' :
             suscripcion.estado === 'activa' ? 'En curso' :
             suscripcion.estado}
          </span>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {suscripcion.patente && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Patente / Dispositivo</h3>
              <p className="text-gray-900 font-semibold">🚗 {suscripcion.patente}</p>
            </div>
          )}
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Tipo</h3>
            <p className="text-gray-900 capitalize">{suscripcion.tipo}</p>
          </div>
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Precio</h3>
            <p className="text-2xl font-bold text-gray-900">${suscripcion.precio.toLocaleString()}</p>
          </div>
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Fecha de Inicio</h3>
            <p className="text-gray-900">
              {(() => {
                const fecha = suscripcion.fecha_inicio.split('T')[0]
                const [year, month, day] = fecha.split('-')
                return `${day}/${month}/${year}`
              })()}
            </p>
          </div>
          {suscripcion.fecha_fin && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Fecha de Fin</h3>
              <p className="text-gray-900">
                {(() => {
                  const fecha = suscripcion.fecha_fin.split('T')[0]
                  const [year, month, day] = fecha.split('-')
                  return `${day}/${month}/${year}`
                })()}
              </p>
            </div>
          )}
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Estado</h3>
            <p className="text-gray-900 capitalize">{suscripcion.estado}</p>
          </div>
        </div>
      </div>

      {/* Modal de confirmación para eliminar */}
      <Modal
        isOpen={modalConfirmar}
        onClose={() => setModalConfirmar(false)}
        title="Eliminar Suscripción"
        message="¿Estás seguro de eliminar esta suscripción? Esta acción no se puede deshacer."
        onConfirm={eliminarSuscripcion}
        confirmText="Eliminar"
        cancelText="Cancelar"
        type="warning"
      />

      {/* Modal de mensaje */}
      <Modal
        isOpen={modalMensaje.isOpen}
        onClose={() => setModalMensaje({ isOpen: false, mensaje: '', tipo: 'success' })}
        title={modalMensaje.tipo === 'success' ? 'Éxito' : 'Error'}
        message={modalMensaje.mensaje}
        type={modalMensaje.tipo}
      />
    </div>
  )
}
